module upcore_tb ();
//------------------------------------------------------------------------------
parameter DATASIZE = 4;
parameter INSTSIZE = 8;
parameter REG_R0 = 2'b00;
parameter REG_R1 = 2'b01;
parameter REG_R2 = 2'b10;
parameter REG_R3 = 2'b11;
parameter STATE_1 = 4'h1;
parameter STATE_2 = 4'h2;
parameter STATE_3 = 4'h4;
parameter STATE_4 = 4'h8;
parameter ALU_ADD = 2'b00;
parameter ALU_SUB = 2'b01;
parameter ALU_AND = 2'b10;
parameter ALU_ORR = 2'b11;
//------------------------------------------------------------------------------
// GROUP-SPECIFIC
//------------------------------------------------------------------------------
parameter CODE_MOV = 2'b01;
parameter CODE_ALU = 2'b10;
parameter CODE_LOD = 2'b11;
parameter CODE_NOP = 2'b00;
//------------------------------------------------------------------------------
// COMMON CODES
//------------------------------------------------------------------------------
// clock generator
parameter CLKP = 10;
reg dclk;
always begin
  #(CLKP/2) dclk = ~dclk;
end
// task: execute inst
reg denb;
reg[INSTSIZE-1:0] dcod;
wire[3:0] mstt;
task exec_inst ;
	input[INSTSIZE-1:0] inst;
	reg[3:0] curr;
	begin
		dcod = inst;
		// from ringctr4_tb
		curr = mstt; denb = 1'b1;
		while (mstt===curr) #(CLKP);
		while (mstt!==curr) #(CLKP);
		denb = 1'b0;
	end
endtask
// task: display selected register
task show_regname ;
	input[1:0] addr;
	begin
		case(addr)
			2'b00: $write("R0");
			2'b01: $write("R1");
			2'b10: $write("R2");
			2'b11: $write("R3");
		endcase
	end
endtask
// task: display selected alu op
task show_alufunc ;
	input[1:0] func;
	begin
		case(func)
			2'b00: $write("add ");
			2'b01: $write("sub ");
			2'b10: $write("and ");
			2'b11: $write("or ");
		endcase
	end
endtask
//------------------------------------------------------------------------------
// GROUP-SPECIFIC: instruction design
//------------------------------------------------------------------------------
// task: execute MOV inst
task exec_mov ;
	input[1:0] dst,src;
	reg[INSTSIZE-1:0] code;
	begin
		code = { CODE_MOV, dst, 2'b00, src }; // 8-b instructions
		$write("[%05g] Exec:{mov ",$time);
		show_regname(dst);
		$write(",");
		show_regname(src);
		$write("} CODE:[%08b]\n",code);
		exec_inst(code);
	end
endtask
// task: execute ALU inst
task exec_alu ;
	input[1:0] sel,op1,op2;
	reg[INSTSIZE-1:0] code;
	begin
		code = { CODE_ALU, op1, sel, op2  };
		$write("[%05g] Exec:{",$time);
		show_alufunc(sel);
		show_regname(op1);
		$write(",");
		show_regname(op2);
		$write("} CODE:[%08b]\n",code);
		exec_inst(code);
	end
endtask
// task: execute LOD inst
task exec_lod ;
	input[1:0] dst;
	input[DATASIZE-1:0] dat;
	reg[INSTSIZE-1:0] code;
	begin
		code = { CODE_LOD, dst, dat };
		$write("[%05g] Exec:{load ",$time);
		show_regname(dst);
		$write(", %04b} CODE:[%08b]\n",dat,code);
		exec_inst(code);
	end
endtask
// task: execute NOP inst
task exec_nop ;
	reg[INSTSIZE-1:0] code;
	begin
		code = { CODE_NOP, 6'b000000 };
		$write("[%05g] Exec:{NOP} CODE:[%08b]\n",$time,code);
		exec_inst(code);
	end
endtask
//------------------------------------------------------------------------------
// COMMON CODES
//------------------------------------------------------------------------------
// task: display register contents
task show_regblock;
  begin
    $write("[%05g] Register Contents => { ",$time);
    $write("R0:%b ",dut.regs.reg0.tdat);
    $write("R1:%b ",dut.regs.reg1.tdat);
    $write("R2:%b ",dut.regs.reg2.tdat);
    $write("R3:%b }\n",dut.regs.reg3.tdat);
  end
endtask
// monitor processor state
reg SHOW_STATE;
always @(mstt) begin // check changes in state
	if (SHOW_STATE===1'b1) begin
		case (mstt)
			STATE_1: $display("[%05g] S1:IDLE (Read Inst)",$time);
			STATE_2: $display("[%05g] S2:R-RD (Read Reg)",$time);
			STATE_3: $display("[%05g] S3:PROC (ALU Ops)",$time);
			STATE_4: $display("[%05g] S4:R-WR (Write Reg)",$time);
			default: $display("[%05g] ??:INVALID (%04b)",$time,mstt);
		endcase
	end
end
// monitor regblock read (S2) / write (S4)
always @(posedge dclk)
begin
	if (mstt==STATE_2) begin
		if (dut.inst_alu||dut.inst_mov) begin
			if (dut.inst_alu) begin
			  $write("[%05g] Read1 from ",$time);
			  show_regname(dut.src1);
			  $display(" (%04b)",dut.opr1);
			end
			$write("[%05g] Read2 from ",$time);
			show_regname(dut.src2);
			$display(" (%04b)",dut.opr2);
		end
		if (dut.inst_lod) begin
			$display("[%05g] Read from xdat (%04b)",$time,dut.ireg_dat[3:0]);
		end
	end
	if (mstt==STATE_3) begin
		if (dut.inst_alu) begin
			$write("[%05g] Processing ( ",$time);
			show_alufunc(dut.alu_sel);
			$display(") => %b @ (%b)",dut.alu_res,dut.alud.tA1);
		end
	end
	if (mstt==STATE_4) begin
		if (dut.d0en) begin
			$write("[%05g] Writing to ",$time);
			show_regname(dut.dst0);
			$display(" (%04b)",dut.xdat);
		end
	end
end
// task: inject value direct into regblock
task hack_regblock;
	input[1:0] addr;
	input[3:0] data;
	begin
		case(addr)
			2'b00: dut.regs.reg0.tdat = data;
			2'b01: dut.regs.reg1.tdat = data; 
			2'b10: dut.regs.reg2.tdat = data;
			2'b11: dut.regs.reg3.tdat = data;
		endcase
	end
endtask
// SPECIAL: function!
// - like task... but can return value!
// - for verification only!
function[3:0] read_regblock;
	input[1:0] addr;
	reg[3:0] data;
	begin
		case(addr)
			2'b00: data = dut.regs.reg0.tdat;
			2'b01: data = dut.regs.reg1.tdat; 
			2'b10: data = dut.regs.reg2.tdat;
			2'b11: data = dut.regs.reg3.tdat;
		endcase
		read_regblock = data;
	end
endfunction
//------------------------------------------------------------------------------
// main testbench code... can differ...
//------------------------------------------------------------------------------
reg drst;
reg[1:0] dest,src1,src2,alop;
reg[3:0] save;
integer loop, next, palu;
initial begin
	SHOW_STATE = 1'b0; // set to 1'b1 to view state transition
	// reset
	dclk = 1'b1; denb = 1'b0;
	drst = 1'b1; // power-on reset
	$display("[%05g] PowerON Reset Init",$time);
	// generate stimuli
	#(CLKP*2) drst = 1'b0; // allow reset state for 2 clk cycle
	$display("[%05g] PowerON Reset Ends",$time);
	show_regblock();
	// test nop
	$display("[%05g] Testing NOP instruction",$time);
	exec_nop();
	show_regblock();
	// test load immediate
	$display("[%05g] Testing LOAD instruction",$time);
	exec_lod(REG_R0,4'ha);
	exec_lod(REG_R1,4'h5);
	exec_lod(REG_R2,4'h9);
	exec_lod(REG_R3,4'hd);
	show_regblock();
	//$stop;
	// test mov
	$display("[%05g] Testing MOVE instruction",$time);
	for (loop=0;loop<4;loop=loop+1) begin
		src1 = loop;
		for (next=0;next<4;next=next+1) begin
			dest = next;
			// hack to restore original value AFTER move completed
			save = read_regblock(dest);
			exec_mov(dest,src1); // dest = src1
			hack_regblock(dest,save);
		end
	end
	show_regblock();
	//$stop;
	// test alu
	for (palu=0;palu<4;palu=palu+1) begin
		alop = ~palu;
		$write("[%05g] Testing ALU @",$time);
		show_alufunc(alop);
		$display("instruction");
		for (loop=0;loop<4;loop=loop+1) begin
			src2 = loop;
			for (next=0;next<4;next=next+1) begin
				dest = next;
				exec_alu(alop,dest,src2); // dest = dest <alop> src2
			end
		end
		show_regblock();
	end
	$stop;
end
wire done;
defparam dut.DATABITS = DATASIZE;
defparam dut.INSTBITS = INSTSIZE;
upcore dut (dclk,drst,denb,dcod,mstt,done);
endmodule
