;----------------------------------------------------------------------
;test nokia 3310 lcd - main program only inits lcd and run simple test
;*3.3v required
;
;*using an 8255 i/o chip in bsr mode (port c)
;- pc0 = sdin
;- pc1 = sclk (positive edge)
;- pc2 = d/c (default low)
;- pc3 = sce (active low)
;- pc4 = rst (active low)
;
;----------------------------------------------------------------------

;----------------------------------------------------------------------
;'environment' constants
;----------------------------------------------------------------------

	cpu "8085.tbl"
portc:	equ 82h
ctrlp:	equ 83h
x_res:	equ 84
y_res:	equ 48 ; not really - divide by 8
csize:	equ 504 ; LCD cache size (x_res*y_res)/8=4032/8

	org 2000h
	jmp init

;----------------------------------------------------------------------
;subroutines
;----------------------------------------------------------------------

;----------------------------------------------------------------------
; lcdchr - display char at current location. counter incremented.
; - acc = character to be sent (retained)
;----------------------------------------------------------------------

lcdchr:	push h
	push d

	push b
	mov c,a ; save char
	sta temp ; save char 'longer'
	; update low watermark?
chkr0:	lhld lmark
	xchg ; store lmark in de
	lhld lcidx
	call cmp16 ; check lcidx < lmark?
	jnc chkr1
	shld lmark ; if yes, lmark = lcidx
chkr1:	mov a,c
	cpi 20h
	jnc chkr2
	jmp cvpch ; convert to printable char
chkr2:	mvi a,7bh
	cmp c
	jnc chkr3
cvpch:	mvi c,5ch
chkr3:	mov a,c
	sbi 20h
	mov c,a
	lxi d,cache
	lhld lcidx
	dad d
	xchg ; de will hold cache index
	mvi h,05h ; index * 5
	mvi b,00h
	mvi a,00h
	inr c ; check if c is zero
	dcr c
	jz gofntx
getfx:	add h
	jnc skipb
	inr b
skipb:	dcr c
	jnz getfx
	mov c,a
gofntx:	lxi h,fontbl
	dad b ; hl will hold font table index
	mvi b,05h
wcache:	mov a,m ; get from table
	xchg
	ora a
	ral ; shift left by 1 - bring in zero?
	mov m,a ; write to cache
	xchg
	inx h
	inx d
	dcr b
	jnz wcache
	; add horizontal gap in cache
	mvi a,00h
	stax d
	; update high watermark?
	lhld lcidx
	lxi d,0005h
	dad d
	shld lcidx
	xchg
	lhld hmark
	call cmp16 ; check if hmark < lcidx
	jnc exchr
	xchg
	shld hmark ; if yes, hmark = lcidx
	inx h
	shld lcidx ; gap increment for index
exchr:	lda temp ; get back char
	pop b
	pop d
	pop h
	ret

;----------------------------------------------------------------------
; lcdgxy - sets 'cursor' location at x & y (base-index = 0)
; - b = x-loc (0 to x_res/6-1), c = y-loc (0 to y_res/8-1)
;----------------------------------------------------------------------

lcdgxy:	push psw
	push h
	push d
	mvi d,00h
	mvi a,00h
	mvi l,84
	inr c ; check if c is zero
	dcr c
	jz goforx
nxtgoy:	add l
	jnc skipd
	inr d
skipd:	dcr c
	jnz nxtgoy
goforx:	mov e,a ; de will have y-index	
	mvi h,00h
	mvi a,00h
	mvi l,6
	inr b ; check if b is zero
	dcr b
	jz goidx
nxtgox:	add l
	jnc skiph
	inr h
skiph:	dcr b
	jnz nxtgox
goidx:	mov l,a ; hl will have x-index
	dad d
	shld lcidx
	pop d
	pop h
	pop psw
	ret

;----------------------------------------------------------------------
; inilcd - initializes lcd and clear the display
;----------------------------------------------------------------------

inilcd:	push h
	push psw ; do we need to preserve psw??
	; reset lcd connections
	mvi a,1fh
	out portc
	call delay
	; reset lcd
	mvi a,08h ; clr bit pc4 (rst)
	out ctrlp
	call delay
	mvi a,09h ; set bit pc4 (rst)
	out ctrlp
	; common initial lcd settings
	call setcmd
	mvi a,21h ; lcd extended commands
	call go_lcd
	mvi a,0c8h ; set lcd vop (contrast)
	call go_lcd
	mvi a,06h ; set temperature coeff
	call go_lcd
	mvi a,13h ; lcd bias mode 1:48
	call go_lcd
	mvi a,20h ; lcd std cmd, horiz addr
	call go_lcd
	mvi a,09h ; all segments on
	call go_lcd
	call delay
	mvi a,08h ; all segments off
	call go_lcd
	call delay
	mvi a,0ch ; lcd normal mode
	call go_lcd
	; clear fist	
	call clrlcd
	call updlcd
	pop psw
	pop h
	ret

;----------------------------------------------------------------------
; clrlcd - clears cache (updlcd must be called for visual effect)
;----------------------------------------------------------------------

clrlcd:	push b
	push h
	push psw ; do we need to preserve psw??
	lxi b,csize
	lxi h,cache
lp_clr:	mvi m,00h
	inx h
	dcx b
	mov a,b
	ora c
	jnz lp_clr
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0000h
	shld lmark
	; updatelcd = true
	mvi a,01h
	sta update
	pop psw
	pop h
	pop b
	ret

;----------------------------------------------------------------------
; updlcd - updates lcd view based on pixel cache memory
;----------------------------------------------------------------------

updlcd:	push b
	push d
	push h
	push psw ; do we need to preserve psw??
	; check low watermark (signed number)
	lhld lmark
	mov a,h
	ral
	jnc clnxt
	lxi h,0000h ; if lmark<0, set as 0
	shld lmark
	jmp setxy
clnxt:	lxi d,csize
	call cmp16
	jc setxy
	lxi h,csize-1 ; if lmark>=csize, set as csize-1
	shld lmark
	; set xy base address based on lmark
setxy:	mvi d,x_res
	call div168
	call setcmd
	mvi a,80h
	ora l
	call go_lcd
	mvi a,40h
	ora e
	call go_lcd
	; check high watermark (signed number)
	lhld hmark
	mov a,h
	ral
	jnc chnxt
	lxi h,0000h ; if hmark<0, set as 0
	shld hmark
	jmp setupd
chnxt:	lxi d,csize
	call cmp16
	jc setupd
	lxi h,csize-1 ; if hmark>=csize, set as csize-1
	shld hmark
	; serialize video buffer
setupd:	xchg ; store hmark in de first
	lhld lmark
	xchg ; now hmark in hl and lmark in de
	call sub16
	inx h
	mov b,h
	mov c,l
	lxi h,cache
	dad d ; offset by lmark
	call setdat
lp_upd:	mov a,m
	call go_lcd
	inx h
	dcx b
	mov a,b
	ora c
	jnz lp_upd
	; reset watermark pointers
	lxi h,csize-1
	shld lmark
	lxi h,0000h
	shld hmark
	; updatelcd = false
	mvi a,00h
	sta update
	pop psw
	pop h
	pop d
	pop b
	ret

;----------------------------------------------------------------------
; div168 - routine to divide 16-bit by 8-bit
; - hl = first 16-bit value, d = second non-zero 8-bit value
; - results in e and remainder in hl
; - *note: acc & flag values will be overwritten!!!
;----------------------------------------------------------------------

div168:	mvi e,00h
	mov a,l
nxdiv:	sub d
	jnc mkdiv
	jz mkdiv
	inr h ; checking z flag without compromising acc
	dcr h
	jz exdiv
	dcr h
mkdiv:	inr e
	mov l,a
	jmp nxdiv
exdiv:	ret

;----------------------------------------------------------------------
; sub16 - routine to subtract 16-bit values
; - hl = first 16-bit value, de = second 16-bit value
; - results in hl
; - *note: acc & flag values will be overwritten!!!
;----------------------------------------------------------------------

sub16:	mov a,l
	sub e
	mov l,a
	jnc nxsub
	dcr h
nxsub:	mov a,h
	sub d
	mov h,a
	ret

;----------------------------------------------------------------------
; cmp16 - routine to compare 16-bit values
; - hl = first 16-bit value, de = second 16-bit value
; - results in flags z & c (as in cmp)
; - *note: acc & flag values will be overwritten!!!
;----------------------------------------------------------------------

cmp16:	push b
	mvi b,02h
	mov a,h
	cmp d
	jc excmp
	jnz skcmp
	dcr b
skcmp:	mov a,l
	cmp e
	jc excmp
	rnz
	; if zero, depends on prev compare
	dcr b ; mod z flag without mod cy flag!
excmp:	pop b	
	ret

;----------------------------------------------------------------------
; setcmd - sets next byte send as command
;----------------------------------------------------------------------

setcmd:	push psw
	mvi a,04h ; clear bit pc2 (d/c)
	out ctrlp
	pop psw
	ret

;----------------------------------------------------------------------
; setdat - sets next byte send as data
;----------------------------------------------------------------------

setdat:	push psw
	mvi a,05h ; set bit pc2 (d/c)
	out ctrlp
	pop psw
	ret

;----------------------------------------------------------------------
; go_lcd - sends data to lcd serially through portc
; - acc = data to be sent (retained)
;----------------------------------------------------------------------

go_lcd:	push b
	mov c,a ; save data
	; enable display controller
	mvi a,06h ; clear bit pc3 (sce)
	out ctrlp
	mvi b,8 ; 8-bit data transfer
lp_go:	mov a,c
	rlc	; send msb first
	mov c,a
	ani 01h	; set/rst bit pc0 (sdin)
	out ctrlp
	mvi a,02h ; clr bit pc1 (sclk)
	out ctrlp
	mvi a,03h ; set bit pc1 (sclk)
	out ctrlp
	dcr b
	jnz lp_go
	; disable display controller
	mvi a,07h ; set bit pc3 (sce)
	out ctrlp
	mov a,c
	pop b
	ret

;----------------------------------------------------------------------
; delay - basic uncalibrated long delay
;----------------------------------------------------------------------

delay:	push b
	push psw
	lxi b,64000
lp_dly:	dcx b
	mov a,b
	ora c
	jnz lp_dly
	pop psw
	pop b
	ret

;----------------------------------------------------------------------
; snuz - basic short animation delay
;----------------------------------------------------------------------

snuz:	push b
	mvi c,200
lpsnz1:	mvi b,255
lp_snz:	dcr b
	jnz lp_snz
	dcr c
	jnz lpsnz1
	pop b
	ret

;----------------------------------------------------------------------
;system initialization
;----------------------------------------------------------------------

	; init stack
init:	lxi sp,4000h
	; init 8255
	mvi a,82h ; all ports output except portb (just in case)
	out ctrlp
	; init lcd
	call inilcd ; set basic horizontal display, clears lcd

;----------------------------------------------------------------------
;main program
;----------------------------------------------------------------------

	mvi b,00h
	mvi c,00h
	call lcdgxy
	lxi h,msg
	; sends message to lcd
lptxt:	mov a,m
	cpi 00h
	jz next1
	call lcdchr
	call updlcd
	inx h
	jmp lptxt
	
next1:	call delay
	; pre-anim frames
	mvi b,04h ; loop for 4 frames
	lxi h,frme1 ; FRAME 1
lpprex:	push b
	lxi b,csize-84 ; minus bank 0!
	lxi d,cache+84 ; start at bank 1 NOT 0!
lpimgx:	mov a,m
	xchg
	mov m,a
	xchg
	inx h
	inx d
	dcx b
	mov a,b
	ora c
	jnz lpimgx
	push h
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd
	call snuz
	call snuz
	pop h ; HL at NEXT frame
	pop b
	dcr b
	jnz lpprex

	; anim loop
	mvi b,05h ; loop 5 times
lpmain:	push b
	; update display cache
	lxi b,csize-84 ; minus bank 0!
	lxi d,cache+84 ; start at bank 1 NOT 0!
	lxi h,frme5 ; FRAME 5
lpimg1:	mov a,m
	xchg
	mov m,a
	xchg
	inx h
	inx d
	dcx b
	mov a,b
	ora c
	jnz lpimg1
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd
	call snuz
	; update display cache
	lxi b,csize-84 ; minus bank 0!
	lxi d,cache+84 ; start at bank 1 NOT 0!
	lxi h,frme6 ; FRAME 6
lpimg2:	mov a,m
	xchg
	mov m,a
	xchg
	inx h
	inx d
	dcx b
	mov a,b
	ora c
	jnz lpimg2
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd
	call snuz
	pop b
	dcr b
	jnz lpmain

	; post-anim frames
	mvi b,05h ; loop for 5 frames
	lxi h,frme5 ; FRAME 5
lppsty:	push b
	lxi b,csize-84 ; minus bank 0!
	lxi d,cache+84 ; start at bank 1 NOT 0!
lpimgy:	mov a,m
	xchg
	mov m,a
	xchg
	inx h
	inx d
	dcx b
	mov a,b
	ora c
	jnz lpimgy
	push h
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd
	call snuz
	call snuz
	pop h ; HL at next frame
	lxi d,csize-x_res; minus 2 frames to get to original prev
	call sub16 ; HL at current frame
	call sub16 ; HL at prev frame
	pop b
	dcr b
	jnz lppsty

	; final display
	lxi b,csize-84 ; minus bank 0!
	lxi h,cache+84 ; start at bank 1 NOT 0!
lpimgz:	mvi m,00h
	inx h
	dcx b
	mov a,b
	ora c
	jnz lpimgz
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd
	call delay
	lxi b,csize-84 ; minus bank 0!
	lxi d,cache+84 ; start at bank 1 NOT 0!
	lxi h,frme5 ; FRAME 5 - final frame
lpimgd:	mov a,m
	xchg
	mov m,a
	xchg
	inx h
	inx d
	dcx b
	mov a,b
	ora c
	jnz lpimgd
	; reset watermark pointers
	lxi h,csize-1
	shld hmark
	lxi h,0054h ; start at bank 1 NOT 0!
	shld lmark
	; go image!!!!
	call updlcd

exit:	jmp 0000h

;----------------------------------------------------------------------
; data storage
;----------------------------------------------------------------------

	; org????h
fontbl:	dfb 00h, 00h, 00h, 00h, 00h ; sp
	dfb 00h, 00h, 2fh, 00h, 00h ; !
	dfb 00h, 07h, 00h, 07h, 00h ; "
	dfb 14h, 7fh, 14h, 7fh, 14h ; #
	dfb 24h, 2ah, 7fh, 2ah, 12h ; $
	dfb 0c4h, 0c8h, 10h, 26h, 46h ; %
	dfb 36h, 49h, 55h, 22h, 50h ; &
	dfb 00h, 05h, 03h, 00h, 00h ; '
	dfb 00h, 1ch, 22h, 41h, 00h ; (
	dfb 00h, 41h, 22h, 1ch, 00h ; )
	dfb 14h, 08h, 3Eh, 08h, 14h ; *
	dfb 08h, 08h, 3Eh, 08h, 08h ; +
	dfb 00h, 00h, 50h, 30h, 00h ; ,
	dfb 10h, 10h, 10h, 10h, 10h ; -
	dfb 00h, 60h, 60h, 00h, 00h ; .
	dfb 20h, 10h, 08h, 04h, 02h ; /
	dfb 3Eh, 51h, 49h, 45h, 3Eh ; 0
	dfb 00h, 42h, 7Fh, 40h, 00h ; 1
	dfb 42h, 61h, 51h, 49h, 46h ; 2
	dfb 21h, 41h, 45h, 4Bh, 31h ; 3
	dfb 18h, 14h, 12h, 7Fh, 10h ; 4
	dfb 27h, 45h, 45h, 45h, 39h ; 5
	dfb 3Ch, 4Ah, 49h, 49h, 30h ; 6
	dfb 01h, 71h, 09h, 05h, 03h ; 7
	dfb 36h, 49h, 49h, 49h, 36h ; 8
	dfb 06h, 49h, 49h, 29h, 1Eh ; 9
	dfb 00h, 36h, 36h, 00h, 00h ; :
	dfb 00h, 56h, 36h, 00h, 00h ; ;
	dfb 08h, 14h, 22h, 41h, 00h ; <
	dfb 14h, 14h, 14h, 14h, 14h ; =
	dfb 00h, 41h, 22h, 14h, 08h ; >
	dfb 02h, 01h, 51h, 09h, 06h ; ?
	dfb 32h, 49h, 59h, 51h, 3Eh ; @
	dfb 7Eh, 11h, 11h, 11h, 7Eh ; A
	dfb 7Fh, 49h, 49h, 49h, 36h ; B
	dfb 3Eh, 41h, 41h, 41h, 22h ; C
	dfb 7Fh, 41h, 41h, 22h, 1Ch ; D
	dfb 7Fh, 49h, 49h, 49h, 41h ; E
	dfb 7Fh, 09h, 09h, 09h, 01h ; F
	dfb 3Eh, 41h, 49h, 49h, 7Ah ; G
	dfb 7Fh, 08h, 08h, 08h, 7Fh ; H
	dfb 00h, 41h, 7Fh, 41h, 00h ; I
	dfb 20h, 40h, 41h, 3Fh, 01h ; J
	dfb 7Fh, 08h, 14h, 22h, 41h ; K
	dfb 7Fh, 40h, 40h, 40h, 40h ; L
	dfb 7Fh, 02h, 0Ch, 02h, 7Fh ; M
	dfb 7Fh, 04h, 08h, 10h, 7Fh ; N
	dfb 3Eh, 41h, 41h, 41h, 3Eh ; O
	dfb 7Fh, 09h, 09h, 09h, 06h ; P
	dfb 3Eh, 41h, 51h, 21h, 5Eh ; Q
	dfb 7Fh, 09h, 19h, 29h, 46h ; R
	dfb 46h, 49h, 49h, 49h, 31h ; S
	dfb 01h, 01h, 7Fh, 01h, 01h ; T
	dfb 3Fh, 40h, 40h, 40h, 3Fh ; U
	dfb 1Fh, 20h, 40h, 20h, 1Fh ; V
	dfb 3Fh, 40h, 38h, 40h, 3Fh ; W
	dfb 63h, 14h, 08h, 14h, 63h ; X
	dfb 07h, 08h, 70h, 08h, 07h ; Y
	dfb 61h, 51h, 49h, 45h, 43h ; Z
	dfb 00h, 7Fh, 41h, 41h, 00h ; [
	dfb 55h, 2Ah, 55h, 2Ah, 55h ; 3ch - default
	dfb 00h, 41h, 41h, 7Fh, 00h ; ]
	dfb 04h, 02h, 01h, 02h, 04h ; ^
	dfb 40h, 40h, 40h, 40h, 40h ; _
	dfb 00h, 01h, 02h, 04h, 00h ; '
	dfb 20h, 54h, 54h, 54h, 78h ; a
	dfb 7Fh, 48h, 44h, 44h, 38h ; b
	dfb 38h, 44h, 44h, 44h, 20h ; c
	dfb 38h, 44h, 44h, 48h, 7Fh ; d
	dfb 38h, 54h, 54h, 54h, 18h ; e
	dfb 08h, 7Eh, 09h, 01h, 02h ; f
	dfb 0Ch, 52h, 52h, 52h, 3Eh ; g
	dfb 7Fh, 08h, 04h, 04h, 78h ; h
	dfb 00h, 44h, 7Dh, 40h, 00h ; i
	dfb 20h, 40h, 44h, 3Dh, 00h ; j
	dfb 7Fh, 10h, 28h, 44h, 00h ; k
	dfb 00h, 41h, 7Fh, 40h, 00h ; l
	dfb 7Ch, 04h, 18h, 04h, 78h ; m
	dfb 7Ch, 08h, 04h, 04h, 78h ; n
	dfb 38h, 44h, 44h, 44h, 38h ; o
	dfb 7Ch, 14h, 14h, 14h, 08h ; p
	dfb 08h, 14h, 14h, 18h, 7Ch ; q
	dfb 7Ch, 08h, 04h, 04h, 08h ; r
	dfb 48h, 54h, 54h, 54h, 20h ; s
	dfb 04h, 3Fh, 44h, 40h, 20h ; t
	dfb 3Ch, 40h, 40h, 20h, 7Ch ; u
	dfb 1Ch, 20h, 40h, 20h, 1Ch ; v
	dfb 3Ch, 40h, 30h, 40h, 3Ch ; w
	dfb 44h, 28h, 10h, 28h, 44h ; x
	dfb 0Ch, 50h, 50h, 50h, 3Ch ; y
	dfb 44h, 64h, 54h, 4Ch, 44h ; z

frme1:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 02h, 04h, 18h, 60h
	dfb 0C2h, 0Ch, 18h, 0E0h, 03h, 0Eh
	dfb 0F8h, 00h, 00h, 0FFh, 00h, 0F0h
	dfb 1Ch, 07h, 0E0h, 38h, 04h, 0F3h
	dfb 08h, 0E4h, 0F2h, 5Ah, 28h, 24h
	dfb 34h, 10h, 10h, 08h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
frme2:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 20h, 40h, 80h, 00h
	dfb 20h, 0C0h, 80h, 00h, 30h, 0E0h
	dfb 80h, 00h, 00h, 0F0h, 00h, 00h
	dfb 0C0h, 70h, 00h, 80h, 40h, 30h
	dfb 80h, 40h, 20h, 0A0h, 80h, 40h
	dfb 40h, 00h, 00h, 80h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 01h, 86h
	dfb 0FCh, 80h, 01h, 0Eh, 00h, 80h
	dfb 8Fh, 80h, 00h, 0Fh, 00h, 8Fh
	dfb 81h, 0C0h, 4Eh, 43h, 40h, 4Fh
	dfb 0C0h, 7Eh, 4Fh, 0C5h, 82h, 02h
	dfb 03h, 01h, 01h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
frme3:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 08h, 10h, 60h, 80h
	dfb 08h, 30h, 60h, 80h, 0Ch, 38h
	dfb 0E0h, 00h, 00h, 0FCh, 00h, 0C0h
	dfb 70h, 1Ch, 80h, 0E0h, 10h, 0CCh
	dfb 20h, 90h, 0C8h, 68h, 0A0h, 90h
	dfb 0D0h, 40h, 40h, 20h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 0C0h, 0E1h
	dfb 0FFh, 0E0h, 0C0h, 43h, 0C0h, 0E0h
	dfb 0E3h, 0E0h, 0C0h, 43h, 40h, 63h
	dfb 20h, 30h, 13h, 10h, 10h, 53h
	dfb 30h, 9Fh, 13h, 31h, 60h, 0C0h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 01h, 73h
	dfb 0DBh, 8Bh, 85h, 82h, 81h, 43h
	dfb 03h, 03h, 01h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 20h
	dfb 43h, 44h, 0C4h, 0A0h, 10h, 0Fh
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
frme4:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 10h, 20h, 0C0h, 00h
	dfb 10h, 60h, 0C0h, 00h, 18h, 70h
	dfb 0C0h, 00h, 00h, 0F8h, 00h, 80h
	dfb 0E0h, 38h, 00h, 0C0h, 20h, 98h
	dfb 40h, 20h, 90h, 0D0h, 40h, 20h
	dfb 0A0h, 80h, 80h, 40h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 80h, 0C3h
	dfb 0FEh, 0C0h, 80h, 87h, 80h, 0C0h
	dfb 0C7h, 0C0h, 80h, 87h, 80h, 0C7h
	dfb 40h, 60h, 27h, 21h, 20h, 0A7h
	dfb 60h, 3Fh, 27h, 62h, 0C1h, 81h
	dfb 01h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 03h, 0E7h
	dfb 0B7h, 17h, 0Bh, 04h, 03h, 87h
	dfb 07h, 07h, 03h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 40h
	dfb 86h, 89h, 88h, 40h, 20h, 1Fh
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 01h, 01h, 07h, 0FDh, 29h, 28h
	dfb 0F8h, 28h, 28h, 0F8h, 28h, 94h
	dfb 0FCh, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 01h, 0C7h, 7Ch, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
frme5:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 08h, 10h, 60h, 80h
	dfb 08h, 30h, 60h, 80h, 0Ch, 38h
	dfb 0E0h, 00h, 00h, 0FCh, 00h, 0C0h
	dfb 70h, 1Ch, 80h, 0E0h, 10h, 0CCh
	dfb 20h, 90h, 0C8h, 68h, 0A0h, 90h
	dfb 0D0h, 40h, 40h, 20h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 0C0h, 0E1h
	dfb 0FFh, 0E0h, 0C0h, 43h, 0C0h, 0E0h
	dfb 0E3h, 0E0h, 0C0h, 43h, 40h, 63h
	dfb 20h, 30h, 13h, 10h, 10h, 53h
	dfb 30h, 9Fh, 13h, 31h, 60h, 0C0h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 01h, 73h
	dfb 0DBh, 8Bh, 85h, 82h, 81h, 43h
	dfb 03h, 03h, 01h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 20h
	dfb 43h, 44h, 0C4h, 0A0h, 10h, 0Fh
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 83h, 0FEh, 94h, 94h
	dfb 0FCh, 94h, 94h, 0FCh, 94h, 4Ah
	dfb 7Eh, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 0E3h, 3Eh, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 07h, 0Ch, 08h, 08h
	dfb 08h, 08h, 10h, 0E0h, 00h, 00h
	dfb 00h, 00h, 0E0h, 30h, 18h, 08h
	dfb 08h, 0Ch, 07h, 01h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
frme6:
; Start Bank 1
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 08h, 10h, 60h, 80h
	dfb 08h, 30h, 60h, 80h, 0Ch, 38h
	dfb 0E0h, 00h, 00h, 0FCh, 00h, 0C0h
	dfb 70h, 1Ch, 80h, 0E0h, 10h, 0CCh
	dfb 20h, 90h, 0C8h, 68h, 0A0h, 90h
	dfb 0D0h, 40h, 40h, 20h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 2
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 0E0h, 0F1h
	dfb 0FFh, 0F0h, 0E0h, 23h, 0E0h, 0F0h
	dfb 0F3h, 0F0h, 0E0h, 23h, 20h, 13h
	dfb 10h, 10h, 13h, 10h, 10h, 53h
	dfb 30h, 9Fh, 13h, 31h, 60h, 0C0h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 3
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 71h
	dfb 0D9h, 8Bh, 84h, 82h, 80h, 41h
	dfb 01h, 01h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 20h
	dfb 43h, 44h, 0C4h, 0A0h, 10h, 0Fh
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 4
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 83h, 0FEh, 94h, 94h
	dfb 0FCh, 94h, 94h, 0FCh, 94h, 4Ah
	dfb 7Eh, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 0E3h, 3Eh, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
; Start Bank 5
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 07h, 0Ch, 08h, 08h
	dfb 08h, 08h, 10h, 0E0h, 00h, 00h
	dfb 00h, 00h, 0E0h, 30h, 18h, 08h
	dfb 08h, 0Ch, 07h, 01h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h
	dfb 00h, 00h, 00h, 00h, 00h, 00h

msg:	dfb " PCB Lab 2005",00h

hmark:	dfs 2 ; high watermark? 16-bit signed integer value
lmark:	dfs 2 ; low water mark? 16-bit signed integer value
lcidx:	dfs 2 ; cache index? 16-bit unsigned value
update:	dfs 1 ; boolean update flag
temp:	dfs 1 ; temporary storage
cache:	dfs csize
endcch:

	end

;----------------------------------------------------------------------
; version 1.1 (20050328)
; - core codesize = 557-bytes, 
; - font tablesize = 455-bytes,
; - cache tablesize = 504-bytes,
; - temp datasize = 8-bytes
;----------------------------------------------------------------------